package gov.va.med.mhv.phr.wellnessreminders.decorator;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.displaytag.decorator.TableDecorator;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.phr.decorator.DecoratorUtils;
import gov.va.med.mhv.phr.transfer.WellnessReminder;
import gov.va.med.mhv.phr.transfer.ImpreciseDateComposite;

/**
 * This class is the decorator for the Wellness Reminder summary and detail display.
 */
public class WellnessReminderTableDecorator extends TableDecorator{

    protected static final String BLANK = StringEscapeUtils.escapeHtml("");
    
    protected static String escapedValueOrBlank(String value) {
        return StringUtils.isBlank(value) ? BLANK : escapedValue(value);
    }

    protected static String escapedValue(String value) {
        return StringEscapeUtils.escapeHtml(value);
    }
        
    
    /*
     * This method retrieves the Imprecise due date and formats it
     * into a readable date/time string for display.
     * 
     * @return a String value containing the formatted due Date
     */
    public String getDueDate() {
    	return process();
    }
    
    private String process(){
        ImpreciseDateComposite dueDate = getWellnessReminder().getDueDateComposite();
        
        if (dueDate == null ||
            	((dueDate.getImpreciseValue() == null && dueDate.getValue() == null) ||
                      	 ("".equals(dueDate.getImpreciseValue()) && ("".equals(dueDate.getValue()))))) {
            return "DUE NOW";
        } 
        else {
        	String value = DecoratorUtils.extractValue(dueDate);
            DateFormat sdf = new SimpleDateFormat("MM/dd/yyyy");
        	try{
        		Date obj = sdf.parse(value);
        		Date today = new Date();
        		if(today.after(obj)){
        			return "PAST DUE";
        		}
        		else{
        			return value;
        		}	
        	}
        	catch (ParseException pe){
        		return value;
        	}
        }    	
    }
    	
    
    /*
     * This method retrieves the Imprecise date last done and formats it
     * into a readable date/time string for display.
     * 
     * @return a String value containing the formatted date last done
     */
    public String getDateLastDone() {
        ImpreciseDateComposite dateLastDone = getWellnessReminder().getDateLastDoneComposite();
        if (dateLastDone == null ||
            	((dateLastDone.getImpreciseValue() == null && dateLastDone.getValue() == null) ||
            			("".equals(dateLastDone.getImpreciseValue()) && dateLastDone.getValue() == null))) {
            return "";
        } else {
        	//return DecoratorUtils.formatDateTime(dateLastDone);
        	return DecoratorUtils.extractValue(dateLastDone);
        }
    }    
    
    /*
     * This method returns the name of the subject of the Wellness Reminder
     * 
     * @return a String contaning the subject.
     */
    public String getSubject() {
        return escapedValue(getWellnessReminder().getSubject());
    }
    
    
    /*
     * This method returns the name of the location of the Wellness Reminder
     * 
     * @return a String contaning the location.
     */
    public String getLocation() {
        return escapedValue(getWellnessReminder().getFacilityInfo().getName());
    }
    
    
    /*
     * This method returns the name of the status of the Wellness Reminder
     * 
     * @return a String contaning the status.
     */
    public String getWellnessStatus() {
        return escapedValue(getWellnessReminder().getWellnessStatus());
    }    
       
    
    /*
     * This method retrieves the Wellness Reminder for the selected row in the table.
     * 
     * @return   the WellnessReminder object for the current row
     * @see 	 WellnessReminder
     */
    private WellnessReminder getWellnessReminder() {
    	Object currentRowObject = getCurrentRowObject();
        Precondition.assertAssignableFrom("currentRowObject", currentRowObject,
        		WellnessReminder.class);
        WellnessReminder wellnessReminder = (WellnessReminder) currentRowObject;
        Precondition.assertNotNull("currentRowObject", wellnessReminder);
        return wellnessReminder;
    }
   
	
}
